/*  -------------------------------------------------
 *  Script:     charts.js
 *  Purpose:    Display charts based on data readings
 *  Version:    3.20.0-ai for build 3201
 *  Author:     Mark Crossley
 *  Ammendments:Neil Thomas
 *  Last edit:  13th Aug 2022 by NT
 *  -------------------------------------------------
 *  Changelog
 *  13th Aug 2022
 *  Added accessibility descriptions to all axis
 *  Requires accessibility modue to be loaded
 *  Made tooltips tabular
*/

var chart, config;

var myRanges = {
    buttons: [{
        count: 12,
        type: 'hour',
        text: '12h'
    }, {
        count: 24,
        type: 'hour',
        text: '24h'
    }, {
        count: 2,
        type: 'day',
        text: '2d'
    }, {
        type: 'all',
        text: 'All'
    }],
    inputEnabled: false,
    selected: 1
};

$(document).ready(function () {
    $('.btn').change(function () {
        var myRadio = $('input[name=options]');
        var checkedValue = myRadio.filter(':checked').val();

        doGraph(checkedValue);
    });

    $.ajax({
        url: "/api/graphdata/availabledata.json",
        dataType: "json",
        success: function (result) {
            if (result.Temperature === undefined || result.Temperature.Count == 0) {
                $('#temp').parent().remove();
            }
            if (result.DailyTemps === undefined || result.DailyTemps.Count == 0) {
                $('#dailytemp').parent().remove();
            }
            if (result.Humidity === undefined || result.Humidity.Count == 0) {
                $('#humidity').parent().remove();
            }
            if (result.Solar === undefined || result.Solar.Count == 0) {
                $('#solar').parent().remove();
            }
            if (result.Sunshine === undefined || result.Sunshine.Count == 0) {
                $('#sunhours').parent().remove();
            }
            if (result.AirQuality === undefined || result.AirQuality.Count == 0) {
                $('#airquality').parent().remove();
            }
        }
    });

    var doGraph = function (value) {
        switch (value) {
            case 'temp':
                doTemp();
                break;
            case 'dailytemp':
                doDailyTemp();
                break;
            case 'press':
                doPress();
                break;
            case 'wind':
                doWind();
                break;
            case 'windDir':
                doWindDir();
                break;
            case 'rain':
                doRain();
                break;
            case 'dailyrain':
                doDailyRain();
                break;
            case 'humidity':
                doHum();
                break;
            case 'solar':
                doSolar();
                break;
            case 'sunhours':
                doSunHours();
                break;
            case 'airquality':
                doAirQuality();
                break;
            default:
                doTemp();
                break;
        }

        parent.location.hash = value;
    };

    $.ajax({url: "/api/graphdata/graphconfig.json", success: function (result) {
        config = result;
        var value = parent.location.hash.replace('#', '');
        doGraph(value);
        // set the correct button
        if (value !== '') {
            $('input[name=options]').removeAttr('checked').parent().removeClass('active');
            $('input[name=options][value=' + value + ']').prop('checked', true).click().parent().addClass('active');
        }
    }});
});

var doTemp = function () {
    var freezing = config.temp.units === 'C' ? 0 : 32;
    var options = {
        chart: {
            renderTo: 'chartcontainer',
            type: 'line',
            borderColor: "var(--theme)",
            alignTicks: false,
            zoomType: 'x',
            borderWidth: 0
        },
        title: {style:{color:"var(--d5)"},text: 'Temperature'},
        credits: {enabled: true},
        xAxis: {
            accessibility: {description: "Date of readings" },
            type: 'datetime',
            ordinal: false,
            labels: { style: { color: "var(--d4)"}},
            dateTimeLabelFormats: {
                day: '%e %B',
                week: '%e %b<br/>%Y',
                month: '%b %y',
                year: '%Y'
            }
        },
        yAxis: [{
                // left
                accessibility: { description: "Temperature recorded" },
                title: {style:{color:"var(--d4)"}, text: 'Temperature (°' + config.temp.units + ')'},
                opposite: false,
                showLastLabel: true,
                labels: {
                    align: 'right',
                    x: -5,
                    formatter: function () {
                        return '<span style="fill: ' + (this.value <= freezing ? 'blue' : 'red') + ';">' + this.value + '</span>';
                    }
                },
                plotLines: [{
                        // freezing line
                        value: freezing,
                        color: 'rgb(0, 0, 180)',
                        width: 1,
                        zIndex: 2
                    }]
            }, {
                // right
                title: {style:{color:"var(--d4)"}, text: 'Temperature (°' + config.temp.units + ')'},
                gridLineWidth: 0,
                opposite: true,
                showLastLabel: true,
                linkedTo: 0,
                labels: {
                    align: 'left',
                    x: 5,
                    formatter: function () {
                        return '<span style="fill: ' + (this.value <= freezing ? 'blue' : 'red') + ';">' + this.value + '</span>';
                    }
                }
            }],
        legend: {enabled: true, itemStyle: {color: "var(--d4)"}},
        plotOptions: {
            series: {
                dataGrouping: {
                    enabled: false
                },
                states: {
                    hover: {
                        halo: {
                            size: 5,
                            opacity: 0.25
                        }
                    }
                },
                cursor: 'pointer',
                marker: {
                    enabled: false,
                    states: {
                        hover: {
                            enabled: true,
                            radius: 0.1
                        }
                    }
                }
            },
            line: {lineWidth: 2}
        },
        tooltip: {
            shared: true,
            split: false,
            style: { color: "var(--d4)", fontSize: '6px;'},
			borderColor: "var(--theme)",
            useHTML: true,
            headerFormat: '<div class="tooltipTitle">{point.key}</div><table class="toolTip">',
            valueSuffix: ' °' + config.temp.units,
            valueDecimals: config.temp.decimals,
            pointFormat: '<tr><td><span style="color:{series.color};">\u25CF</span> {series.name}:</td>' +
						 '<td>{point.y}{tooltip.suffix}</td></tr>',
			footerFormat: '</table>',
            xDateFormat: "%A, %b %e, %H:%M"
        },
        series: [],
        rangeSelector: myRanges
    };

    chart = new Highcharts.StockChart(options);
    chart.showLoading();

    $.ajax({
        url: '/api/graphdata/tempdata.json',
        dataType: 'json',
        success: function (resp) {
           var titles = {
               'temp'     : 'Temperature',
               'dew'      : 'Dew Point',
               'apptemp'  : 'Apparent',
               'feelslike': 'Feels Like',
               'wchill'   : 'Wind Chill',
               'heatindex': 'Heat Index',
               'humidex'  : 'Humidex',
               'intemp'   : 'Inside'
            };
            var idxs = ['temp', 'dew', 'apptemp', 'feelslike', 'wchill', 'heatindex', 'humidex', 'intemp'];
            var yaxis = 0;

            idxs.forEach(function(idx) {
                var valueSuffix = ' °' + config.temp.units;
                yaxis = 0;

                if (idx in resp) {
                    if (idx === 'humidex') {
                        valueSuffix = null;
                        if (config.temp.units == 'F') {
                            chart.yAxis[1].remove();
                            chart.addAxis({
                                id: 'humidex',
                                title:{text: 'Humidex'},
                                opposite: true,
                                labels: {
                                    align: 'left'
                                },
                                alignTicks: true,
                                gridLineWidth: 0, // Not working?
                                gridZIndex: -10, // Hides the grid lines for this axis
                                showEmpty: false
                            }, false, false);

                            yaxis = 'humidex';
                        }
                    }

                    chart.addSeries({
                        name: titles[idx],
                        id: 'series-' + idx,
                        data: resp[idx],
                        yAxis: yaxis,
                        tooltip: {valueSuffix: valueSuffix}
                    }, false);

                    if (idx === 'temp') {
                        chart.get('series-' + idx).options.zIndex = 99;
                    }
                }
            });
            chart.hideLoading();
            chart.redraw();
        }
    });
};

var doPress = function () {
    var options = {
        chart: {
            renderTo: 'chartcontainer',
            type: 'line',
            borderWidth: 0,
            alignTicks: false
        },
        title: {style:{color:"var(--d5)"},text: 'Pressure'},
        credits: {enabled: true},
        xAxis: {
            accessibility: {description: "Date & time of reading." }, 
            type: 'datetime',
            ordinal: false,
            labels: { style: { color: "var(--d5)"}},
            dateTimeLabelFormats: {
                day: '%e %b',
                week: '%e %b %y',
                month: '%b %y',
                year: '%Y'
            }
        },
        yAxis: [{
                // left
                accessibility: { description: "Barometric pressure reading." },
                title: {style:{color:"var(--d5)"},text: 'Pressure (' + config.press.units + ')'},
                opposite: false,
                labels: {
                    align: 'right',
                    x: -5,
                    style: { color: "var(--d4)" }
                }
            }, {
                // right
                linkedTo: 0,
                gridLineWidth: 0,
                opposite: true,
                title: {text: null},
                labels: {
                    align: 'left',
                    x: 5,
                    style: { color: "var(--d4)" }
                }
            }],
        legend: {enabled: true, itemStyle: {color: "var(--d4)"}},
        plotOptions: {
            series: {
                dataGrouping: {
                    enabled: false
                },
                states: {
                    hover: {
                        halo: {
                            size: 5,
                            opacity: 0.25
                        }
                    }
                },
                cursor: 'pointer',
                marker: {
                    enabled: false,
                    states: {
                        hover: {
                            enabled: true,
                            radius: 0.1
                        }
                    }
                }
            },
            line: {lineWidth: 2}
        },
        tooltip: {
            shared: true,
            split: false,
            style: { color: "var(--d4)", fontSize: '6px;'},
            borderColor: "var(--theme)",
            useHTML: true,
            headerFormat: '<div class="tooltipTitle">{point.key}</div><table class="toolTip">',
            pointFormat: '<tr><td><span style="color:{series.color};">\u25CF</span> {series.name}:</td>' +
						 '<td>{point.y}{tooltip.suffix}</td></tr>',
			footerFormat: '</table>',
            valueSuffix: ' ' + config.press.units,
            valueDecimals: config.press.decimals,
            xDateFormat: "%A, %b %e, %H:%M"
        },
        series: [{
                name: 'Pressure'
            }],
        rangeSelector: myRanges
    };

    chart = new Highcharts.StockChart(options);
    chart.showLoading();

    $.ajax({
        url: '/api/graphdata/pressdata.json',
        dataType: 'json',
        success: function (resp) {
            chart.hideLoading();
            chart.series[0].setData(resp.press);
        }
    });
};

var compassP = function (deg) {
    var a = ['N', 'NE', 'E', 'SE', 'S', 'SW', 'W', 'NW'];
    return a[Math.floor((deg + 22.5) / 45) % 8];
};

var doWindDir = function () {
    var options = {
        chart: {
            renderTo: 'chartcontainer',
            type: 'scatter',
            borderColor: "var(--theme)",
            alignTicks: false
        },
        title: {style:{color:"var(--d5)"},text: 'Wind Direction'},
        credits: {enabled: true},
        boost: {
            useGPUTranslations: false,
            usePreAllocated: true
        },
        xAxis: {
            accessibility: { description: "Date & time of reading." },
            type: 'datetime',
            ordinal: false,
            labels: { style: { color: "var(--d5)"}},
            dateTimeLabelFormats: {
                day: '%e %b',
                week: '%e %b %y',
                month: '%b %y',
                year: '%Y'
            }
        },
        yAxis: [{
                // left
                accessibility: { description: "Wind bearing." },
                title: {style:{color:"var(--d5)"},text: 'Bearing'},
                opposite: false,
                min: 0,
                max: 360,
                tickInterval: 45,
                labels: {
                    align: 'right',
                    x: -5,
                    style: { color: "var(--d4)" },
                    formatter: function () {
                        return compassP(this.value);
                    }
                }
            }, {
                // right
                accessibility: { description: "Wind compass point." },
                linkedTo: 0,
                gridLineWidth: 0,
                opposite: true,
                title: {text: null},
                min: 0,
                max: 360,
                tickInterval: 45,
                labels: {
                    align: 'left',
                    x: 5,
                    formatter: function () {
                        return compassP(this.value);
                    },
                    style: { color: "var(--d4)" }
                }
            }],
        legend: {enabled: true, itemStyle: {color: "var(--d4)"}},
        plotOptions: {
            scatter: {
                animationLimit: 1,
                cursor: 'pointer',
                enableMouseTracking: false,
                boostThreshold: 200,
                marker: {
                    states: {
                        hover: {enabled: false},
                        select: {enabled: false},
                        normal: {enabled: false}
                    }
                },
                shadow: false,
                label: {enabled: false}
            }

        },
        tooltip: {
            enabled: false
        },
        series: [{
                name: 'Bearing',
                type: 'scatter',
                marker: {
                    symbol: 'circle',
                    radius: 2
                }
            }, {
                name: 'Avg Bearing',
                type: 'scatter',
                color: 'red',
                marker: {
                    symbol: 'circle',
                    radius: 2
                }
            }
        ],
        rangeSelector: myRanges
    };

    chart = new Highcharts.StockChart(options);
    chart.showLoading();

    $.ajax({
        url: '/api/graphdata/wdirdata.json',
        dataType: 'json',
        success: function (resp) {
            chart.hideLoading();
            chart.series[0].setData(resp.bearing);
            chart.series[1].setData(resp.avgbearing);
        }
    });
};

var doWind = function () {
    var options = {
        chart: {
            renderTo: 'chartcontainer',
            type: 'line',
            borderWidth: 0,
            alignTicks: false
        },
        title: {style:{color:"var(--d5)"},text: 'Wind Speed'},
        credits: {enabled: true},
        xAxis: {
            accessibility: { description: "Date and time of reading." },
            type: 'datetime',
            ordinal: false,
            labels: { style: { color: "var(--d5)"}},
            dateTimeLabelFormats: {
                day: '%e %b',
                week: '%e %b %y',
                month: '%b %y',
                year: '%Y'
            }
        },
        yAxis: [{
                // left
                accessibility: { description: "Wind speed recorded in " + config.wind.units },
                title: {style:{color:"var(--d5)"},text: 'Wind Speed (' + config.wind.units + ')'},
                opposite: false,
                min: 0,
                labels: {
                    align: 'right',
                    x: -5,
                    style: { color: "var(--d4)" }
                }
            }, {
                // right
                linkedTo: 0,
                gridLineWidth: 0,
                opposite: true,
                min: 0,
                title: {text: null},
                labels: {
                    align: 'left',
                    x: 5,
                    style: { color: "var(--d4)" }
                }
            }],
        legend: {enabled: true, itemStyle: {color: "var(--d4)"}},
        plotOptions: {
            series: {
                dataGrouping: {
                    enabled: false
                },
                states: {
                    hover: {
                        halo: {
                            size: 5,
                            opacity: 0.25
                        }
                    }
                },
                cursor: 'pointer',
                marker: {
                    enabled: false,
                    states: {
                        hover: {
                            enabled: true,
                            radius: 0.1
                        }
                    }
                }
            },
            line: {lineWidth: 2}
        },
        tooltip: {
            shared: true,
            split: false,
			style: { color: "var(--d4)", fontSize: '6px;'},
            borderColor: "var(--theme)",
            useHTML: true,
            headerFormat: '<div class="tooltipTitle">{point.key}</div><table class="toolTip">',
            pointFormat: '<tr><td><span style="color:{series.color};">\u25CF</span> {series.name}:</td>' +
						 '<td>{point.y}{tooltip.suffix}</td></tr>',
			footerFormat: '</table>',
            valueSuffix: ' ' + config.wind.units,
            valueDecimals: config.wind.decimals,
            xDateFormat: "%A, %b %e, %H:%M"
        },
        series: [{
                name: 'Wind Speed'
            }, {
                name: 'Wind Gust'
        }],
        rangeSelector: myRanges
    };

    chart = new Highcharts.StockChart(options);
    chart.showLoading();

    $.ajax({
        url: '/api/graphdata/winddata.json',
        dataType: 'json',
        success: function (resp) {
            chart.hideLoading();
            chart.series[0].setData(resp.wspeed);
            chart.series[1].setData(resp.wgust);
        }
    });
};

var doRain = function () {
    var options = {
        chart: {
            renderTo: 'chartcontainer',
            type: 'line',
            borderWidth: 0,
            alignTicks: true
        },
        title: { style: { color: "var(--d5)"}, text: 'Rainfall'},
        credits: {enabled: true},
        xAxis: {
            accessibility: { description: "Date and time of reading." },
            type: 'datetime',
            ordinal: false,
            labels: { style: { color: "var(--d5)"}},
            dateTimeLabelFormats: {
                day: '%e %b',
                week: '%e %b %Y',
                month: '%b %y',
                year: '%Y'
            }
        },
        yAxis: [{
                // left
                accessibility: { description: "Rainfall rate" },
                title: {style:{color:"var(--d5)"},text: 'Rainfall rate (' + config.rain.units + '/hr)'},
                min: 0,
                opposite: false,
                labels: {
                    align: 'right',
                    x: -5,
                    style: { color: "var(--d4)" }
                },
                showEmpty: false
            }, {
                // right
                accessibility: { description: "Total cumuluative rainfall." },
                opposite: true,
                title: {style:{color:"var(--d5)"},text: 'Rainfall (' + config.rain.units + ')'},
                min: 0,
                labels: {
                    align: 'left',
                    x: 5,
                    style: { color: "var(--d4)" }
                }
            }],
        legend: {enabled: true, itemStyle: {color: "var(--d4)"}},
        plotOptions: {
            series: {
                boostThreshold: 0,
                dataGrouping: {
                    enabled: false
                },
                showInNavigator: true,
                states: {
                    hover: {
                        halo: {
                            size: 5,
                            opacity: 0.25
                        }
                    }
                },
                cursor: 'pointer',
                marker: {
                    enabled: false,
                    states: {
                        hover: {
                            enabled: true,
                            radius: 0.1
                        }
                    }
                }
            },
            line: {lineWidth: 2}
        },
        navigator: { yAxis: { min: 0 }},
        tooltip: {
            shared: true,
            split: false,
            borderColor: "var(--theme)",
            style: { color: "var(--d4)", fontSize: '6px;'},
            useHTML: true,
            headerFormat: '<div class="tooltipTitle">{point.key}</div><table class="tooltip">',
            pointFormat: '<tr><td><span style="color:{series.color};">\u25CF</span> {series.name}:</td>' +
						 '<td>{point.y}{tooltip.suffix}</td></tr>',
			footerFormat: '</table>',
            valueDecimals: config.rain.decimals,
            xDateFormat: "%A, %b %e, %H:%M"
        },
        series: [{
                name: 'Daily rain',
                type: 'area',
                yAxis: 1,
                tooltip: {valueSuffix: ' ' + config.rain.units},
                fillOpacity: 0.3
            }, {
                name: 'Rain rate',
                type: 'line',
                yAxis: 0,
                tooltip: {valueSuffix: ' ' + config.rain.units + '/hr'}
        }],
        rangeSelector: myRanges
    };

    chart = new Highcharts.StockChart(options);
    chart.showLoading();

    $.ajax({
        url: '/api/graphdata/raindata.json',
        dataType: 'json',
        success: function (resp) {
            chart.hideLoading();
            chart.series[0].setData(resp.rfall);
            chart.series[1].setData(resp.rrate);
        }
    });
};

var doHum = function () {
    var options = {
        chart: {
            renderTo: 'chartcontainer',
            type: 'line',
            borderWidth: 0,
            alignTicks: false
        },
        title: {style:{color:"var(--d5)"},text: 'Relative Humidity'},
        credits: {enabled: true},
        xAxis: {
            accessibility: { description: "Date and time of reading." },
            type: 'datetime',
            ordinal: false,
            labels: { style: { color: "var(--d5)"}},
            dateTimeLabelFormats: {
                day: '%e %b',
                week: '%e %b %y',
                month: '%b %y',
                year: '%Y'
            }
        },
        yAxis: [{
                // left
                accessibility: { description: "Relative humidity" },
                title: {style:{color:"var(--d4)"},text: 'Humidity (%)'},
                opposite: false,
                min: 0,
                max: 100,
                labels: {
                    align: 'right',
                    x: -5,
                    style: { color: "var(--d4)" }
                }
            }, {
                // right
                linkedTo: 0,
                gridLineWidth: 0,
                opposite: true,
                min: 0,
                max: 100,
                title: {text: null},
                labels: {
                    align: 'left',
                    x: 5,
                    style: { color: "var(--d4)" }
                }
            }],
        legend: {enabled: true, itemStyle: {color: "var(--d4)"}},
        plotOptions: {
            series: {
                dataGrouping: {
                    enabled: false
                },
                states: {
                    hover: {
                        halo: {
                            size: 5,
                            opacity: 0.25
                        }

                    }
                },
                cursor: 'pointer',
                marker: {
                    enabled: false,
                    states: {
                        hover: {
                            enabled: true,
                            radius: 0.1
                        }
                    }
                }
            },
            line: {lineWidth: 2}
        },
        tooltip: {
            shared: true,
            split: false,
            borderColor: "var(--theme)",
            style: { color: "var(--d4)", fontSize: '6px;'},
            useHTML: true,
            headerFormat: "<div class='tooltipTitle'>{point.key}</div><table class='toolTip'>",
            pointFormat: '<tr><td><span style="color:{series.color};">\u25CF</span> {series.name}:</td>' +
						 '<td>{point.y}{tooltip.suffix}</td></tr>',
			footerFormat: '</table>',
            valueSuffix: '%',
            valueDecimals: config.hum.decimals,
            xDateFormat: "%A, %b %e, %H:%M"
        },
        series: [],
        rangeSelector: myRanges
    };

    chart = new Highcharts.StockChart(options);
    chart.showLoading();

    $.ajax({
        url: '/api/graphdata/humdata.json',
        dataType: 'json',
        success: function (resp) {
            var titles = {
                'hum'  : 'Outdoor Humidity',
                'inhum': 'Indoor Humidity'
             }
             var idxs = ['hum', 'inhum'];
             var cnt = 0;
             idxs.forEach(function(idx) {
                 if (idx in resp) {
                     chart.addSeries({
                         name: titles[idx],
                         data: resp[idx]
                     }, false);

                     cnt++;
                 }
             });

            chart.hideLoading();
            chart.redraw();
        }
    });
};

var doSolar = function () {
    var options = {
        chart: {
            renderTo: 'chartcontainer',
            type: 'line',
            borderWidth: 0,
            alignTicks: true
        },
        title: {style:{color:"var(--d5)"},text: 'Solar'},
        credits: {enabled: true},
        xAxis: {
            accessibility: { description: "Date and time of reading." },
            type: 'datetime',
            ordinal: false,
            labels: { style: { color: "var(--d4)"}},
            dateTimeLabelFormats: {
                day: '%e %b',
                week: '%e %b %y',
                month: '%b %y',
                year: '%Y'
            }
        },
        yAxis: [],
        legend: {enabled: true, itemStyle: {color: "var(--d4)"}},
        plotOptions: {
            series: {
                boostThreshold: 0,
                dataGrouping: {
                    enabled: false
                },
                showInNavigator: true,
                states: {
                    hover: {
                        halo: {
                            size: 5,
                            opacity: 0.25
                        }

                    }
                },
                cursor: 'pointer',
                marker: {
                    enabled: false,
                    states: {
                        hover: {
                            enabled: true,
                            radius: 0.1
                        }
                    }
                }
            },
            line: {lineWidth: 2}
        },
        tooltip: {
            shared: true,
            split: false,
            borderColor: "var(--theme)",
            style: { color: "var(--d4)", fontSize: '6px;'},
            useHTML: true,
            headerFormat: '<div class="tooltipTitle">{point.key}</div><table class="toolTip">',
            pointFormat: '<tr><td><span style="color:{series.color};">\u25CF</span> {series.name}:</td>' +
						 '<td>{point.y}{tooltip.suffix}</td></tr>',
			footerFormat: '</table>',
            xDateFormat: "%A, %b %e, %H:%M"
        },
        series: [],
        rangeSelector: myRanges
    };

    chart = new Highcharts.StockChart(options);
    chart.showLoading();

    $.ajax({
        url: '/api/graphdata/solardata.json',
        dataType: 'json',
        success: function (resp) {
            var titles = {
                SolarRad       : 'Solar Radiation',
                CurrentSolarMax: 'Theoretical Max',
                UV: 'UV Index'
            };
            var types = {
                SolarRad: 'area',
                CurrentSolarMax: 'area',
                UV: 'line'
            };
            var colours = {
                SolarRad: 'rgb(255,165,0)',
                CurrentSolarMax: 'rgb(128,128,128)',
                UV: 'rgb(0,0,255)'
            };
            var tooltips = {
                SolarRad: {
                    valueSuffix: ' W/m\u00B2',
                    valueDecimals: 0
                },
                CurrentSolarMax: {
                    valueSuffix: ' W/m\u00B2',
                    valueDecimals: 0
                },
                UV: {
                    valueSuffix: null,
                    valueDecimals: config.uv.decimals
                }
            };

            var idxs = ['SolarRad', 'CurrentSolarMax', 'UV'];
            var cnt = 0;
            var solarAxisCreated = false;

            idxs.forEach(function(idx) {
                if (idx in resp) {
                    if (idx === 'UV') {
                        chart.addAxis({
                            id: 'uv',
                            accessibility: { description: "Date and time of UV Reading." }, 
                            title:{style:{color:"var(--d4)"},text: 'UV Index'},
                            opposite: true,
                            min: 0,
                            labels: {
                                align: 'left',
								style: { color: "var(--d4)" }
                            }
                        });
                    } else if (!solarAxisCreated) {
                        chart.addAxis({
                            id: 'solar',
                            accessibility: { description: "Date and time of 'Solar' reading." },
                            title: {style:{color:"var(--d4)"},text: 'Solar Radiation (W/m\u00B2)'},
                            min: 0,
                            opposite: false,
                            labels: {
                                align: 'right',
                                x: -5,
								style: { color: "var(--d4)" }
                            },
                            showEmpty: false
                        });
                        solarAxisCreated = true;
                    }

                    chart.addSeries({
                        name: titles[idx],
                        type: types[idx],
                        yAxis: idx === 'UV' ? 'uv' : 'solar', accessibility: { description: "Solar radiation." },
                        tooltip: tooltips[idx],
                        data: resp[idx],
                        color: colours[idx],
                        fillOpacity: idx === 'CurrentSolarMax' ? 0.2 : 0.5,
                        zIndex: 100 - cnt
                    }, false);

                    cnt++;
                }
            });

            chart.hideLoading();
            chart.redraw();
        }
    });
};

var doSunHours = function () {
    var options = {
        chart: {
            renderTo: 'chartcontainer',
            type: 'column',
            borderWidth: 0,
            alignTicks: false
        },
        title: {style:{color:"var(--d5)"},text: 'Sunshine Hours'},
        credits: {enabled: true},
        xAxis: {
            accessibility: { description: "Date of reading." },
            type: 'datetime',
            ordinal: false,
            labels: { style: { color: "var(--d4)"}},
            dateTimeLabelFormats: {
                day: '%e %b',
                week: '%e %b %y',
                month: '%b %y',
                year: '%Y'
            }
        },
        yAxis: [{
                // left
                accessibility: { description: "Hours of sunshine recorded." },
                title: {style:{color:"var(--d4)"},text: 'Sunshine Hours'},
                min: 0,
                opposite: false,
                labels: {
                    align: 'right',
                    x: -12,
                    style: { color: "var(--d4)" }
                }
            }, {
                // right
                linkedTo: 0,
                gridLineWidth: 0,
                opposite: true,
                title: {text: null},
                labels: {
                    align: 'left',
                    x: 12,
                    style: { color: "var(--d4)" }
                }
            }],
        legend: {enabled: true, itemStyle: {color: "var(--d4)"}},
        plotOptions: {
            series: {
                dataGrouping: {
                    enabled: false
                },
                pointPadding: 0,
                groupPadding: 0.1,
                states: {
                    hover: {
                        halo: {
                            size: 5,
                            opacity: 0.25
                        }
                    }
                },
                cursor: 'pointer',
                marker: {
                    enabled: false,
                    states: {
                        hover: {
                            enabled: true,
                            radius: 0.1
                        }
                    }
                }
            }
        },
        tooltip: {
            shared: true,
            split: false,
            borderColor: "var(--theme)",
            style: { color: "var(--d4)", fontSize: '6px;'},
            useHTML: true,
            headerFormat: '<div class="tooltipTitle">{point.key}</div><table class="toolTip">',
            pointFormat: '<tr><td><span style="color:{series.color};">\u25CF</span> {series.name}:</td>' +
						 '<td>{point.y}{tooltip.suffix}</td></tr>',
			footerFormat: '</table>',
            xDateFormat: "%A, %b %e"
        },
        series: [{
                name: 'Sunshine Hours',
                type: 'column',
                color: 'rgb(255,165,0)',
                yAxis: 0,
                valueDecimals: 1,
                tooltip: {valueSuffix: ' Hrs'}
            }]
    };

    chart = new Highcharts.Chart(options);
    chart.showLoading();

    $.ajax({
        url: '/api/graphdata/sunhours.json',
        dataType: 'json',
        success: function (resp) {
            chart.hideLoading();
            chart.series[0].setData(resp.sunhours);
        }
    });
};

var doDailyRain = function () {
    var options = {
        chart: {
            renderTo: 'chartcontainer',
            type: 'column',
            borderWidth: 0,
            alignTicks: false
        },
        title: {style:{color:"var(--d5)"},text: 'Daily Rainfall'},
        credits: {enabled: true},
        xAxis: {
            accessibility: { description: "Date of readings" },
            type: 'datetime',
            ordinal: false,
            labels: { style: { color: "var(--d4)"}},
            dateTimeLabelFormats: {
                day: '%e %b',
                week: '%e %b %y',
                month: '%b %y',
                year: '%Y'
            }
        },
        yAxis: [{
                // left
                accessibility: { description: "Total rain recorded for that day." },
                title: {style:{color:"var(--d4)"},text: 'Daily Rainfall'},
                min: 0,
                opposite: false,
                labels: {
                    align: 'right',
                    x: -12,
                    style: { color: "var(--d4)" }
                }
            }, {
                // right
                linkedTo: 0,
                gridLineWidth: 0,
                opposite: true,
                title: {text: null},
                labels: {
                    align: 'left',
                    x: 12,
                    style: { color: "var(--d4)" }
                }
            }],
        legend: {enabled: true, itemStyle: {color: "var(--d4)"}},
        plotOptions: {
            series: {
                dataGrouping: {
                    enabled: false
                },
                pointPadding: 0,
                groupPadding: 0.1,
                states: {
                    hover: {
                        halo: {
                            size: 5,
                            opacity: 0.25
                        }

                    }
                },
                cursor: 'pointer',
                marker: {
                    enabled: false,
                    states: {
                        hover: {
                            enabled: true,
                            radius: 0.1
                        }
                    }
                }
            }
        },
        tooltip: {
            shared: true,
            split: false,
            borderColor: "var(--theme)",
            style: { color: "var(--d4)", fontSize: '6px;'},
            useHTML: true,
            headerFormat: '<div class="tooltipTitle">{point.key}</div><table class="toolTip">',
            pointFormat: '<tr><td><span style="color:{series.color};">\u25CF</span> {series.name}:</td>' +
						 '<td>{point.y}{tooltip.suffix}</td></tr>',
			footerFormat: '</table>',
            xDateFormat: "%A, %b %e"
        },
        series: [{
                name: 'Daily Rainfall',
                type: 'column',
                color: 'blue',
                yAxis: 0,
                valueDecimals: config.rain.decimals,
                tooltip: {valueSuffix: ' ' + config.rain.units}
            }]
    };

    chart = new Highcharts.Chart(options);
    chart.showLoading();

    $.ajax({
        url: '/api/graphdata/dailyrain.json',
        dataType: 'json',
        success: function (resp) {
            chart.hideLoading();
            chart.series[0].setData(resp.dailyrain);
        }
    });
};

var doDailyTemp = function () {
    var freezing = config.temp.units === 'C' ? 0 : 32;
    var options = {
        chart: {
            renderTo: 'chartcontainer',
            type: 'line',
            borderWidth: 0,
            alignTicks: false
        },
        title: {style:{color:"var(--d5)"},text: 'Daily Temperature'},
        credits: {enabled: true},
        xAxis: {
            accessibility: { description: "Date of reading." },
            type: 'datetime',
            ordinal: false,
            labels: { style: { color: "var(--d4)"}},
            dateTimeLabelFormats: {
                day: '%e %b',
                week: '%e %b %y',
                month: '%b %y',
                year: '%Y'
            }
        },
        yAxis: [{
                // left
                accessibility: { description: "Temperatures recorded." },
                title: {style:{color:"var(--d4)"},text: 'Daily Temperature (°' + config.temp.units + ')'},
                opposite: false,
                labels: {
                    align: 'right',
                    x: -5,
                    formatter: function () {
                        return '<span style="fill: ' + (this.value <= 0 ? 'blue' : 'red') + ';">' + this.value + '</span>';
                    }
                },
                plotLines: [{
                        // freezing line
                        value: freezing,
                        color: 'rgb(0, 0, 180)',
                        width: 1,
                        zIndex: 2
                    }]
            }, {
                // right
                linkedTo: 0,
                gridLineWidth: 0,
                opposite: true,
                title: {text: null},
                labels: {
                    align: 'left',
                    x: 5,
                    formatter: function () {
                        return '<span style="fill: ' + (this.value <= 0 ? 'blue' : 'red') + ';">' + this.value + '</span>';
                    }
                }
            }],
        legend: {enabled: true, itemStyle: {color: "var(--d4)"}},
        plotOptions: {
            series: {
                dataGrouping: {
                    enabled: false
                },
                states: {
                    hover: {
                        halo: {
                            size: 5,
                            opacity: 0.25
                        }

                    }
                },
                cursor: 'pointer',
                marker: {
                    enabled: false,
                    states: {
                        hover: {
                            enabled: true,
                            radius: 0.1
                        }
                    }
                }
            },
            line: {lineWidth: 2}
        },
        tooltip: {
            shared: true,
            split: false,
            borderColor: "var(--theme)",
            style: { color: "var(--d4)", fontSize: '6px;'},
            useHTML: true,
            headerFormat: '<div class="tooltipTitle">{point.key}</div><table class="toolTip">',
            pointFormat: '<tr><td><span style="color:{series.color};">\u25CF</span> {series.name}:</td>' +
						 '<td>{point.y}{tooltip.suffix}</td></tr>',
			footerFormat: '</table>',
            valueSuffix: ' °' + config.temp.units,
            valueDecimals: config.temp.decimals,
            xDateFormat: "%A, %b %e"
        },
        rangeSelector: {
            enabled: false
        },
        series: []
    };

    chart = new Highcharts.StockChart(options);
    chart.showLoading();

    $.ajax({
        url: '/api/graphdata/dailytemp.json',
        dataType: 'json',
        success: function (resp) {
            var titles = {
                'avgtemp': 'Avg Temp',
                'mintemp': 'Min Temp',
                'maxtemp': 'Max Temp'
            };
            var colours = {
                'avgtemp': 'green',
                'mintemp': 'blue',
                'maxtemp': 'red'
            };
            var idxs = ['avgtemp', 'mintemp', 'maxtemp'];

            idxs.forEach(function (idx) {
                if (idx in resp) {
                    chart.addSeries({
                        name: titles[idx],
                        data: resp[idx],
                        color: colours[idx]
                    }, false);
                }
            });

            chart.hideLoading();
            chart.redraw();
        }
    });
};

var doAirQuality = function () {
    var options = {
        chart: {
            renderTo: 'chartcontainer',
            type: 'line',
            borderColor: "var(--theme)",
            alignTicks: false
        },
        title: {style:{color:"var(--d5)"},text: 'Air Quality'},
        credits: {enabled: true},
        xAxis: {
            accessibility: { description: "Date and time of reading." },
            type: 'datetime',
            ordinal: false,
            labels: { style: { color: "var(--d5)"}},
            dateTimeLabelFormats: {
                day: '%e %b',
                week: '%e %b %y',
                month: '%b %y',
                year: '%Y'
            }
        },
        yAxis: [{
                // left
                accessibility: { description: "Recorded ppm for particles recorded." },
                title: {style:{color:"var(--d5)"},text: 'µg/m³'},
                opposite: false,
                min: 0,
                labels: {
                    align: 'right',
                    x: -5,
                    style: { color: "var(--d4)" }
                }
            }, {
                // right
                linkedTo: 0,
                gridLineWidth: 0,
                opposite: true,
                min: 0,
                title: {text: null},
                labels: {
                    align: 'left',
                    x: 5,
                    style: { color: "var(--d4)" }
                }
            }],
        legend: {enabled: true, itemStyle: {color: "var(--d4)"}},
        plotOptions: {
            series: {
                dataGrouping: {
                    enabled: false
                },
                states: {
                    hover: {
                        halo: {
                            size: 5,
                            opacity: 0.25
                        }

                    }
                },
                cursor: 'pointer',
                marker: {
                    enabled: false,
                    states: {
                        hover: {
                            enabled: true,
                            radius: 0.1
                        }
                    }
                }
            },
            line: {lineWidth: 2}
        },
        tooltip: {
            shared: true,
            split: false,
            borderColor: "var(--theme)",
            style: { color: "var(--d4)"},
            style: { color: "var(--d4)", fontSize: '6px;'},
            useHTML: true,
            headerFormat: '<div class="tooltipTitle">{point.key}</div><table class="toolTip">',
            pointFormat: '<tr><td><span style="color:{series.color};">\u25CF</span> {series.name}:</td>' +
						 '<td>{point.y}{tooltip.suffix}</td></tr>',
            valueSuffix: ' µg/m³',
            valueDecimals: 1,
            xDateFormat: "%A, %b %e, %H:%M"
        },
        series: [],
        rangeSelector: myRanges
    };

    chart = new Highcharts.StockChart(options);
    chart.showLoading();

    $.ajax({
        url: '/api/graphdata/airqualitydata.json',
        dataType: 'json',
        success: function (resp) {
            var titles = {
                'pm2p5': 'PM 2.5',
                'pm10' : 'PM 10'
             }
             var idxs = ['pm2p5', 'pm10'];
             idxs.forEach(function(idx) {
                 if (idx in resp) {
                     chart.addSeries({
                         name: titles[idx],
                         data: resp[idx]
                     }, false);
                 }
             });

            chart.hideLoading();
            chart.redraw();
        }
    });
};

