/*	~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * 	Script:	page-manager.js
 * 	Role:		Utility for all pages:
 * 	a)	Setting default configuration
 * 	b)	Setting top and bottom margins
 * 	c)	Setting the required theme
 * 	d)	Providing utility functions
 * 	Version:	3.0.0
 * 	Author:	Neil Thomas
 * 	Edited:	27-04-2023 14:17
 * 	Checked:
 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 *	Global settings
 *	These values are used if local string storage is NOT available
 *	They can be edited directly but please note that theme names
 * 	use the filename without the prefix 'thm'.
*/

let CMXConfig = { 'Theme': 'DefaultBlue',
				  'darkMode': false,
				  'StaticHead': true,
				  'StaticFoot': true,
				  'PaddingTop': '10px',
				  'PaddingBottom': '120px'
				 }

let menu = mobileMenu = "";
let initialLoad = true;
let load_menu 	= "js/menu_new.js"; // menu file to load - path is relative to the page

let cmx_data;


let dataLoadedPromise = new Promise((myResolve, myReject) => {
	$( function() {
		setTheme();
		getPageData(myResolve, myReject);
	});
});

let setTheme = function() {
	var mode = "";
	if( CMXConfig.darkMode ) { var mode = '-Dark'; }
	$('#theme').attr('href', 'css/thm' + CMXConfig.Theme + mode + '.css' );
};

let getPageData = function (resolve, reject) {
	$.getJSON('websitedata.json?_=' + Date.now(), function (json) {
		console.log('Data success');
		// auto update every 60 seconds, only the index and today pages
		// Some sites may have index.htm as the default page, and thus not have a page name
		let pageName = window.location.href.split('/').pop().split('.')[0];
		if (pageName == 'index' || pageName == 'today' || pageName == 'todayyest' || pageName == '') {
			setTimeout(function () {
				getPageData(null, null);
			}, 60 * 1000);
		}

		cmx_data = json;

		// Set some header stuff
		var curTitle = $(document).prop('title');
		$(document).prop('title', cmx_data.location + ' - ' + curTitle);
		$('meta[name=description]').attr('content', cmx_data.location + ' weather data');
		$('meta[name=keywords]').attr('content', $('meta[name=keywords]').attr('content') + ', ' + cmx_data.location);

		// do the menus
		if (initialLoad) {
			$.getScript(load_menu, function() { // path is relative to the page - allows for multiple vars to be available and ignores comments ;-)
				createMainMenu(menuSrc, false);
				createMobileMenu(menuSrc, false);
				setupPage();
			});
			initialLoad = false;
		}

		// Show/hide Apparent/Feels Like
		if (cmx_data.options.useApparent === "1") {
			$('[data-cmx-apparent]').removeClass('w3-hide');
			$('[data-cmx-feels]').addClass('w3-hide');
		} 

		if (cmx_data.options.showSolar === "1") {
			$('[data-cmx-solar]').removeClass('w3-hide');
		} else {
			$('[data-cmx-solar-gauge]').addClass('w3-hide'); // Gauges do not draw correctly if hidden from the start
		}

		if (cmx_data.options.showUV === "1") {
			$('[data-cmx-uv]').removeClass('w3-hide');
		} else {
			$('[data-cmx-uv-gauge]').addClass('w3-hide'); // Gauges do not draw correctly if hidden from the start
		}

		// Update all spans having data-cmxdata with data values
		$('[data-cmxdata]').each(function () {
			this.innerHTML = cmx_data[this.dataset.cmxdata];
		});

		if (cmx_data.currcond != '') {
			$('#currCond').removeClass('w3-hide');
		}

		// Use this to trigger other scripts on the page
		if (null !== resolve) {
			resolve();
		}
	})
	.fail(function (jqxhr, textStatus, error) {
		let err = textStatus + ', ' + error;
		console.log('Data Request Failed: ' + err );

		if (null !== reject) {
			reject();
		}

		// lets try that again
		setTimeout(function () {
			getPageData(resolve, reject);
		}, 5000);
	});
};

let createMainMenu = function(src, submenu) {
	let classMain = 'w3-bar-item w3-btn ow-theme-hvr ow-slim w3-hide-small w3-hide-medium';
	let classMainSub = 'w3-bar-item w3-btn ow-theme-sub5-hvr ow-slim';

	src.forEach(function(itm) {
		if (itm.menu !== 'n') {	// wanted in main menu
			if (itm.submenu) { // drop down
				menu += '<div class="w3-dropdown-hover">\n';
				menu += '\t<button id="' +  itm.title.replace(/ /g,"_") + '" type="button" class="w3-btn ow-theme-hvr ow-slim w3-hide-medium w3-hide-small" onclick="dropDown(this)" aria-expanded="false">' + itm.title + '&#8230;</button>\n';
				menu += '\t<div id="sub_' +  itm.title.replace(/ /g,"_") + '" class="w3-dropdown-content w3-bar-block ow-theme">\n';
				// add the sub-menu items
				createMainMenu(itm.items, true);
				menu += '\t\n</div></div>\n';
			} else {
				infill = (itm.new_window ? ' target="_blank"' : '');

				if (itm.forum) {
					 if (cmx_data.forumurl != '') {
						menu += '<a href="' + cmx_data.forumurl + '"' + infill + ' class="' + (submenu ? classMainSub : classMain) + '">' + itm.title + '</a>\n';
					 }
				} else if (itm.webcam) {
					if (cmx_data.webcamurl != '') {
						menu += '<a href="' + cmx_data.webcamurl + '"' + infill + ' class="' + (submenu ? classMainSub : classMain) + '">' + itm.title + '</a>\n';
					}
				} else {
					menu += '<a href="' + itm.url + '"' + infill + ' class="' + (submenu ? classMainSub : classMain) + '">' + itm.title + '</a>\n';
				}
			}
		}
	});

	// if we are processing a sub menu, return to the main loop
	if (submenu)
		return;

	menu += '<button class="w3-bar-item w3-btn ow-theme-hvr ow-slim w3-hide-large w3-right" onClick="toggleMenu(\'Main_Menu_Mobile\')">Menu  &#9776;</button>';

	// stick the menus into the page
	$('#Main_Menu').html(menu);
};

let createMobileMenu = function(src, submenu) {
	let classMobile = 'w3-bar-item w3-btn ow-theme-hvr ow-slim';
	let styleMobile = submenu ? 'padding-left: 30px !important;' : '';

	src.forEach(function(itm) {
		if (itm.menu !== 'w') { // wanted in narrow menu
		// mobile menu
			if (itm.submenu) {
				mobileMenu += '\t<button class="w3-btn ow-slim" style="cursor: default;">' + itm.title + '&#8230;</button>\n';
				createMobileMenu(itm.items, true);
			} else {
				infill = (itm.new_window ? " target='_blank'" : "");
				if (itm.forum && cmx_data.forumurl != '') {
					mobileMenu += '<a href="' +  cmx_data.forumurl + '"' + infill + ' class="' + classMobile + '" style= "' + styleMobile + '">' + itm.title + '</a>\n';
				} else if (itm.webcam && cmx_data.webcamurl != '') {
					mobileMenu += '<a href="' +  cmx_data.webcamurl + '"' + infill + ' class="' + classMobile + '" style= "' + styleMobile + '">' + itm.title + '</a>\n';
				} else {
					mobileMenu += '<a href="' +  itm.url + '"' + infill + ' class="' + classMobile + '" style= "' + styleMobile + '">' + itm.title + '</a>\n';
				}
			}
		}
	});

	// if we are processing a sub menu, return to the main loop
	if (submenu)
		return;

	// stick the menus into the page
	$('#Main_Menu_Mobile').html(mobileMenu);
};


let setupPage = function() {
	//	Adjust page margins and padding
	var elHeight;
	if( CMXConfig.StaticHead ) {
		$('header').addClass( 'w3-top' );
		elHeight = $('header').outerHeight( false );
		$('#content').css('margin-top', elHeight + 'px');
	} else {
		$('header').removeClass('w3-top');
		$('#content').css('margin-top', '0px');
	}
	if( CMXConfig.StaticFoot && ($(window).width() > 600) ) {
		$('footer').addClass('w3-bottom');
		elHeight = $('footer').outerHeight( true );
		$('#content').css('margin-bottom', elHeight + 'px' );
	} else {
		$('footer').removeClass('w3-bottom');
		$('#content').css('margin-bottom', '0px');
	}
	//	Set padding
	$('#content').css('padding-top', CMXConfig.PaddingTop);
	$('#content').css('padding-bottom', CMXConfig.PaddingBottom);
	//	Set Content height
	elHeight = $('header').outerHeight( true ) + $('footer').outerHeight( true ); 
	var contentHeight = $(window).height() - elHeight + 'px';
	//console.log(" Content height: " + contentHeight ) 
	$('#content').css('min-height', contentHeight );
	 var davisPanel = localStorage.getItem('CMXDavis');
	 if( davisPanel === 'Hide' ) {
		$('#DavisPanel').addClass('w3-hide');
		$('#DavisSet').text('Show Davis Info');
	}
};

let showModal = function(modal){
	$('#' + modal).toggleClass('w3-show', 'w3-hide');
	if( $('#' + modal).hasClass('w3-show') ) {
		var child = $('#' + modal).children('.w3-modal-content');
		var mHeight = $(child).outerHeight( true );
		$( child ).css('margin-top', ($(window).outerHeight() - mHeight) / 3 +'px');
	}
};

window.onresize = function() {
	setupPage();
}

let toggleTabs = function( elm, cls ) {
	//	Hides or reveals single panel tabs
	$("." + cls).addClass("w3-hide");
	$("#" + elm).removeClass("w3-hide");
};

let toggleMenu = function(menuId) {
	//	Assumes currently hidden
	console.log("Toggling menu: " + menuId);
	$('#' + menuId).toggleClass( 'w3-show', 'w3-hide' );
};

let toggleArea = function(area) {
	$('#' + area + 'Panel').toggleClass("w3-hide", "w3-show");
	var btnText = $('#' + area + 'Set').text();
	if( btnText.slice(0,4) == "Show") {
		$('#' + area + 'Set').text(btnText.replace("Show","Hide"));
		if( area == 'Davis' ) {
			localStorage.setItem('CMXDavis', 'Show');
		}
	} else {
		$('#' + area + 'Set').text(btnText.replace("Hide","Show"));
		if( area == 'Davis' ) {
			localStorage.setItem( 'CMXDavis', 'Hide' );
		}
	}
		
};

